;;; Enable use of ERRSET function:
(nodebug)
(defparameter test-number 0)
(defparameter bugs-found 0)

(defvar *abort-on-error* nil)	; Abort tests on error, otherwise continue
(defvar *verbose-tests* t)	; Print message on successful tests too.

;;; Define a central error-report fn:
(defun r-fail (i x why)
  (setq bugs-found (1+ bugs-found))
  (format t "***** Failed test number ~s: ~s~%  because: ~a~%" i x why)
  (when *abort-on-error* (top-level)))

(defun passed-test (i x) 
       (when *verbose-tests* 
	     (format t "Passed test ~s: ~s~%" i x)))

;;; Squeeze out some redundancy:
(defmacro should-fail (x)
  (setq test-number (1+ test-number))
  `(if (errset ,x nil) 
       (r-fail test-number ',x "execution didn't fail")
       (passed-test test-number ',x)))
(defmacro should-work (x)
  (setq test-number (1+ test-number))
  `(if (not (errset ,x nil)) 
       (r-fail test-number ',x "execution failed")
       (passed-test test-number ',x)))
(defmacro should-be-true (x &aux (y (gensym)))
  (setq test-number (1+ test-number))
  `(let ((,y (errset ,x nil))) 
       (cond ((null ,y) (r-fail test-number ',x "execution failed"))
	     ((eq (car ,y) 'nil) (r-fail test-number ',x "returned NIL"))
	     (t (passed-test test-number ',x)))))
(defmacro should-be-false (x &aux (y (gensym)))
  (setq test-number (1+ test-number))
  `(let ((,y (errset ,x nil))) 
       (cond ((null ,y) (r-fail test-number ',x "execution failed"))
	     ((not (eq (car ,y) 'nil)) 
	      (r-fail test-number ',x (format nil "returned ~s" (car ,y))))
	     (t (passed-test test-number ',x)))))


; TESTS GO HERE

; check coerce on sequence conversions
(should-be-true (eq nil (coerce #() 'cons)))
(should-be-true (eq nil (coerce #() 'list)))	; synonym
(should-be-true (eq nil (coerce "" 'cons)))
(should-be-true (equal '(1 2) (coerce #(1 2) 'cons)))
(should-be-true (equal '(#\a #\b) (coerce "ab" 'cons)))
(should-be-true (eq nil (coerce (coerce nil 'array) 'cons)))
(should-be-true (eq nil (coerce (coerce "" 'array) 'cons)))
(should-be-true (equal '(#\a #\b) (coerce (coerce "ab" 'array) 'cons)))
(should-be-true (equal '(1 2) (coerce (coerce '(1 2) 'array) 'cons)))
(should-be-true (equal "" (coerce nil 'string)))
(should-be-true (equal "" (coerce #() 'string)))
(should-be-true (equal "ab" (coerce '(#\a #\b) 'string)))
(should-be-true (equal "ab" (coerce #(#\a #\b) 'string)))

(defun eqv (a b) 
       (and (arrayp a)
	    (arrayp b)
	    (equal (coerce a 'cons) (coerce b 'cons))))

(defun identity (x) x)
; concatenate

(should-be-true (eq nil (concatenate 'cons)))
(should-be-true (eq nil (concatenate 'list)))	; synonym
(should-be-true (equal '(#\a #\b c d e f)
		       (concatenate 'cons "ab" '(c d) #(e f))))
(should-be-true (equal "" (concatenate 'string)))
(should-be-true (equal "abcdef"
		       (concatenate 'string "ab" '(#\c #\d) #(#\e #\f))))
(should-be-true (eq nil (coerce (concatenate 'array) 'cons)))
(should-be-true (eqv #(#\a #\b c d e f)
		     (concatenate 'array "ab" '(c d) #(e f))))	
(should-fail (concatenate 'fixnum))
(should-fail (concatenate 'cons 'x))
(should-fail (concatenate 'cons 7))


; elt

(setq x '(nil nil nil))
(should-be-true (eq (setf (elt x 0) 'a) 'a))
(should-be-true (eq (setf (elt x 2) 'c) 'c))
(should-fail (setf (elt x 3) 'd))
(should-fail (setf (elt x -1) 'd))
(should-be-true (eq (elt x 0) 'a))
(should-be-true (eq (elt x 2) 'c))
(should-be-true (equal x '(a nil c)))
(should-fail (elt x 3))
(should-fail (elt x -1))


(setq x #(nil nil nil))
(should-be-true (eq (setf (elt x 0) 'a) 'a))
(should-be-true (eq (setf (elt x 2) 'c) 'c))
(should-fail (setf (elt x 3) 'd))
(should-fail (setf (elt x -1) 'd))
(should-be-true (eq (elt x 0) 'a))
(should-be-true (eq (elt x 2) 'c))
(should-be-true (eqv x #(a nil c)))
(should-fail (elt x 3))
(should-fail (elt x -1))


(setq x "XXX")
(should-be-true (eq (setf (elt x 0) #\a) #\a))
(should-be-true (eq (setf (elt x 2) #\c) #\c))
(should-fail (setf (elt x 3) #\d))
(should-fail (setf (elt x -1) #\d))
(should-be-true (eq (elt x 0) #\a))
(should-be-true (eq (elt x 2) #\c))
(should-be-true (equal x "aXc"))
(should-fail (elt x 3))
(should-fail (elt x -1))


; Test MAP
(should-fail (map 'cons #'(lambda () nil)))
(should-be-true (eq (map 'cons #'identity #()) nil))
(should-be-true (eqv (map 'array #'identity "") #()))
(should-be-true (equal (map 'string #'identity nil) ""))
(should-be-true (equal (map 'cons #'list '(a b c z) #(d e f) "ghi")
		       '((a d #\g) (b e #\h) (c f #\i))))
(should-be-true (equal (map 'cons #'list '(a b c) #(d e f z) "ghi")
		       '((a d #\g) (b e #\h) (c f #\i))))
(should-be-true (equal (map 'cons #'list '(a b c) #(d e f) "ghiz")
		       '((a d #\g) (b e #\h) (c f #\i))))

(should-be-true (eqv (map 'array #'list '(a b c z) #(d e f) "ghi")
		     #((a d #\g) (b e #\h) (c f #\i))))
(should-be-true (eqv (map 'array #'list '(a b c) #(d e f z) "ghi")
		     #((a d #\g) (b e #\h) (c f #\i))))
(should-be-true (eqv (map 'array #'list '(a b c) #(d e f) "ghiz")
		     #((a d #\g) (b e #\h) (c f #\i))))

(should-be-true (equal (map 'string #'(lambda (a b c) 
					      (when (and (eq a b)(eq a c))
						    a))
			    '(#\a #\b #\c #\d) #(#\a #\b #\c) "abc")
		       "abc"))

(setq x nil)
(should-be-true (eq (map nil #'(lambda (y) (setq x (cons y x)))
			 '(a b c d))
		    nil))
(should-be-true (equal x '(d c b a)))


; Test every, notevery, some, notany
; These are implemented with same code

(should-fail (every #'cons))
(should-be-true (every #'evenp #(2 4 6 8)))
(should-be-true (every #'characterp "abcde"))
(should-be-true (every #'> '(2 4 6 8) #(1 3 5 7)))
(should-be-false (every #'> '(2 4 6 8) #(1 3 6 7)))
(should-be-true (every #'evenp '(2 4 6 8)))
(should-be-true (notevery #'evenp '(2 4 6 9)))
(should-be-false (notevery #'evenp '(2 4 6 8)))
(should-be-true (some #'evenp '(2 3 5 7)))
(should-be-false (some #'evenp '(1 3 5 7)))
(should-be-false (notany #'evenp '(2 3 5 7)))
(should-be-true (notany #'evenp '(1 3 5 7)))

; Test length

(should-be-true (zerop (length nil)))
(should-be-true (zerop (length "")))
(should-be-true (zerop (length #())))
(should-be-true (eql (length '(a b c)) 3))
(should-be-true (eql (length #(a b c)) 3))
(should-be-true (eql (length "abc") 3))

; Test reverse, nreverse
(should-be-true (equal (reverse nil) nil))
(should-be-true (equal (reverse '(a b c d)) '(d c b a)))
(should-be-true (equal (nreverse nil) nil))
(should-be-true (equal (nreverse '(a b c d)) '(d c b a)))
(should-be-true (eqv (reverse #()) #()))
(should-be-true (eqv (reverse #(a b c d)) #(d c b a)))
(should-be-true (eqv (nreverse #()) #()))
(should-be-true (eqv (nreverse #(a b c d)) #(d c b a)))
(should-be-true (equal (reverse "") ""))
(should-be-true (equal (reverse "abcd") "dcba"))
(should-be-true (equal (nreverse "") ""))
(should-be-true (equal (nreverse "abcd") "dcba"))

; Test subseq
(should-be-true (eq (subseq '(a b c) 3) nil))
(should-fail (subseq '(a b c) 4))
(should-fail (subseq '(a b c) -1))
(should-be-true (equal (subseq '(a b c) 0) '( a b c)))
(should-be-true (equal (subseq '(a b c) 1) '(b c)))
(should-be-true (equal (subseq '(a b c) 0 0) nil))
(should-be-true (equal (subseq '(a b c) 1 1) nil))
(should-be-true (equal (subseq '(a b c) 2 nil) '(c)))
(should-be-true (equal (subseq '(a b c) 1 2) '(b)))
(should-fail (subseq '(a b c) 2 1))
(should-be-true (equal (subseq nil 0 0) nil))

(should-be-true (eqv (subseq #(a b c) 3) #()))
(should-fail (subseq #(a b c) 4))
(should-fail (subseq #(a b c) -1))
(should-be-true (eqv (subseq #(a b c) 0) #( a b c)))
(should-be-true (eqv (subseq #(a b c) 1) #(b c)))
(should-be-true (eqv (subseq #(a b c) 0 0) #()))
(should-be-true (eqv (subseq #(a b c) 1 1) #()))
(should-be-true (eqv (subseq #(a b c) 2 nil) #(c)))
(should-be-true (eqv (subseq #(a b c) 1 2) #(b)))
(should-fail (subseq #(a b c) 2 1))
(should-be-true (eqv (subseq #() 0 0) #()))

(should-be-true (equal (subseq "abc" 3) ""))
(should-fail (subseq "abc" 4))
(should-fail (subseq "abc" -1))
(should-be-true (equal (subseq "abc" 0) "abc"))
(should-be-true (equal (subseq "abc" 1) "bc"))
(should-be-true (equal (subseq "abc" 0 0) ""))
(should-be-true (equal (subseq "abc" 1 1) ""))
(should-be-true (equal (subseq "abc" 2 nil) "c"))
(should-be-true (equal (subseq "abc" 1 2) "b"))
(should-fail (subseq "abc" 2 1))
(should-be-true (equal (subseq "" 0 0) ""))


; search
(should-be-true (eql (search "abc" "dabc") 1))
(should-be-true (eql (search "xabcx" "dabcq" 
			     :start1 1 :end1 4 :start2 1 :end2 4) 1))
(should-be-true (eql (search "xabcx" "dabcq" 
			     :start1 1 :end1 5 :start2 1 :end2 4) nil))
(should-be-true (eql (search "xabcx" "dabcq" 
			     :start1 1 :end1 4 :start2 1) 1))
(should-be-true (eql (search "a" "dcba" :end2 3) nil))
(should-be-true (eql (search "b" "dcba" :end2 3) 2))
(should-be-true (eql (search "abcx" "abababc" :end1 3) 4))
(should-be-true (eql (search "abcx" "abababc" :end1 2) 0))
(should-be-true (eql (search "abcx" "abababc" :end1 2 :start2 1) 2))
(should-be-true (eql (search "bax" "abababc" :end1 2 :start2 1) 1))
(should-be-true (eql (search "xxx" "xx") nil))
(should-be-true (eql (search "xxx" "xxx") 0))
(should-be-true (eql (search '(a b c) '(d a b c)) 1))
(should-be-true (eql (search "" "xxx") 0))
(should-be-true (eql (search "" "xxx" :start2 1) 1))
(should-be-true (eql (search "" "") nil))
(should-be-true (eql (search "a" "a" :start1 1 :start2 1) nil))
(should-be-true (eql (search "a" "") nil))


(should-be-true (eql (search '(x a b c x) '(d a b c q) 
			     :start1 1 :end1 4 :start2 1 :end2 4) 1))
(should-be-true (eql (search '(x a b c x) '(d a b c q) 
			     :start1 1 :end1 5 :start2 1 :end2 4) nil))
(should-be-true (eql (search '(x a b c x) '(d a b c q) 
			     :start1 1 :end1 4 :start2 1) 1))
(should-be-true (eql (search '(a) '(d c b a) :end2 3) nil))
(should-be-true (eql (search '(b) '(d c b a) :end2 3) 2))
(should-be-true (eql (search '(a b c x) '(a b a b a b c) :end1 3) 4))
(should-be-true (eql (search '(a b c x) '(a b a b a b c) :end1 2) 0))
(should-be-true (eql (search '(a b c x) '(a b a b a b c) :end1 2 :start2 1) 2))
(should-be-true (eql (search '(b a x) '(a b a b a b c) :end1 2 :start2 1) 1))
(should-be-true (eql (search '(x x x) '(x x)) nil))
(should-be-true (eql (search '(x x x) '(x x x)) 0))
(should-be-true (eql (search '() '(x x x)) 0))
(should-be-true (eql (search '() '(x x x) :start2 1) 1))
(should-be-true (eql (search '() '()) nil))
(should-be-true (eql (search '(a) '(a) :start1 1 :start2 1) nil))
(should-be-true (eql (search '(a) '()) nil))

(should-be-true (eql (search #(x a b c x) #(d a b c q) 
			     :start1 1 :end1 4 :start2 1 :end2 4) 1))
(should-be-true (eql (search #(x a b c x) #(d a b c q) 
			     :start1 1 :end1 5 :start2 1 :end2 4) nil))
(should-be-true (eql (search #(x a b c x) #(d a b c q) 
			     :start1 1 :end1 4 :start2 1) 1))
(should-be-true (eql (search #(a) #(d c b a) :end2 3) nil))
(should-be-true (eql (search #(b) #(d c b a) :end2 3) 2))
(should-be-true (eql (search #(a b c x) #(a b a b a b c) :end1 3) 4))
(should-be-true (eql (search #(a b c x) #(a b a b a b c) :end1 2) 0))
(should-be-true (eql (search #(a b c x) #(a b a b a b c) :end1 2 :start2 1) 2))
(should-be-true (eql (search #(b a x) #(a b a b a b c) :end1 2 :start2 1) 1))
(should-be-true (eql (search #(x x x) #(x x)) nil))
(should-be-true (eql (search #(x x x) #(x x x)) 0))
(should-be-true (eql (search #() #(x x x)) 0))
(should-be-true (eql (search #() #(x x x) :start2 1) 1))
(should-be-true (eql (search #() #()) nil))
(should-be-true (eql (search #(a) #(a) :start1 1 :start2 1) nil))
(should-be-true (eql (search #(a) #()) nil))


(should-be-true (eql (search #(b a) '(c b a)) 1))
(should-be-true (eql (search '(#\b #\a) "cba") 1))
(should-be-true (eql (search "ba" #(#\c #\b #\a)) 1))

(should-be-true (eql (search '((b) (a)) '((c) (b) (a)) :test #'equal) 1))
(should-be-true (eql (search '((b) (a)) '((c) (b) (a)) :key #'car) 1))
(should-be-true (eql (search '((c) (b)) '((c) (b) (a)) :test-not #'equal) 1))

(should-fail (search "abc" "abc" :start1 -1))
(should-fail (search "abc" "abc" :start1 4))
(should-fail (search "abc" "abc" :end1 4))
(should-fail (search "abc" "abc" :end1 -1))
(should-fail (search "abc" "abc" :start1 2 :end1 1))
(should-be-true (eql (search "b" "abc" :end1 nil) 1))

;; nsubstitute and substitute
(should-be-true (equal (nsubstitute 'x 'b '()) '()))
(should-be-true (equal (nsubstitute 'x 'b '(a b a b a)) '(a x a x a)))
(should-be-true (equal (nsubstitute 'x 'a '(a b a b a)) '(x b x b x)))
(should-be-true (equal (nsubstitute 'x 'b '(a b a b a) :start 1 :end 4)
		       '(a x a x a)))
(should-be-true (equal (nsubstitute 'x 'a '(a b a b a) :start 1 :end 4)
		       '(a b x b a)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count -1)
		       '(b a c b d b)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 0)
		       '(b a c b d b)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 1 :from-end nil)
		       '(x a c b d b)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 2)
		       '(x a c x d b)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 3)
		       '(x a c x d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 4)
		       '(x a c x d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count nil)
		       '(x a c x d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 0 :from-end t)
		       '(b a c b d b)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 1 :from-end t)
		       '(b a c b d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 2 :from-end t)
		       '(b a c x d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 3 :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count 4 :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :count nil :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (nsubstitute 'x 'b '(b a c b d b) :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (nsubstitute 'x 'a '(a a a a a a) :start 1 :end 4 :count 2)
		       '(a x x a a a)))
(should-be-true (equal (nsubstitute 'x 'a '(a a a a a a) :start 1 :end 4 :count 5)
		       '(a x x x a a)))
(should-be-true (equal (nsubstitute 'x 'a '(a a a a a a)
				    :start 1 :end 4 :count 2 :from-end t)
		       '(a a x x a a)))
(should-be-true (equal (nsubstitute 'x 'a '(a a a a a a) :start 1 :end 4 :count 5)
		       '(a x x x a a)))

(should-be-true (eqv (nsubstitute 'x 'b #()) #()))
(should-be-true (eqv (nsubstitute 'x 'b #(a b a b a)) #(a x a x a)))
(should-be-true (eqv (nsubstitute 'x 'a #(a b a b a)) #(x b x b x)))
(should-be-true (eqv (nsubstitute 'x 'b #(a b a b a) :start 1 :end 4)
		       #(a x a x a)))
(should-be-true (eqv (nsubstitute 'x 'a #(a b a b a) :start 1 :end 4)
		       #(a b x b a)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count -1)
		       #(b a c b d b)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 0)
		       #(b a c b d b)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 1 :from-end nil)
		       #(x a c b d b)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 2)
		       #(x a c x d b)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 3)
		       #(x a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 4)
		       #(x a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count nil)
		       #(x a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 0 :from-end t)
		       #(b a c b d b)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 1 :from-end t)
		       #(b a c b d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 2 :from-end t)
		       #(b a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 3 :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count 4 :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :count nil :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'b #(b a c b d b) :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (nsubstitute 'x 'a #(a a a a a a) :start 1 :end 4 :count 2)
		       #(a x x a a a)))
(should-be-true (eqv (nsubstitute 'x 'a #(a a a a a a) :start 1 :end 4 :count 5)
		       #(a x x x a a)))
(should-be-true (eqv (nsubstitute 'x 'a #(a a a a a a)
				    :start 1 :end 4 :count 2 :from-end t)
		       #(a a x x a a)))
(should-be-true (eqv (nsubstitute 'x 'a #(a a a a a a) :start 1 :end 4 :count 5)
		       #(a x x x a a)))

(should-be-true (equal (nsubstitute #\x #\b "") ""))
(should-be-true (equal (nsubstitute #\x #\b "ababa") "axaxa"))
(should-be-true (equal (nsubstitute #\x #\a "ababa") "xbxbx"))
(should-be-true (equal (nsubstitute #\x #\b "ababa" :start 1 :end 4)
		       "axaxa"))
(should-be-true (equal (nsubstitute #\x #\a "ababa" :start 1 :end 4)
		       "abxba"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count -1)
		       "bacbdb"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 0)
		       "bacbdb"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 1 :from-end nil)
		       "xacbdb"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 2)
		       "xacxdb"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 3)
		       "xacxdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 4)
		       "xacxdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count nil)
		       "xacxdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 0 :from-end t)
		       "bacbdb"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 1 :from-end t)
		       "bacbdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 2 :from-end t)
		       "bacxdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 3 :from-end t)
		       "xacxdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count 4 :from-end t)
		       "xacxdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :count nil :from-end t)
		       "xacxdx"))
(should-be-true (equal (nsubstitute #\x #\b "bacbdb" :from-end t)
		       "xacxdx"))
(should-be-true (equal (nsubstitute #\x #\a "aaaaaa" :start 1 :end 4 :count 2)
		       "axxaaa"))
(should-be-true (equal (nsubstitute #\x #\a "aaaaaa" :start 1 :end 4 :count 5)
		       "axxxaa"))
(should-be-true (equal (nsubstitute #\x #\a "aaaaaa"
				    :start 1 :end 4 :count 2 :from-end t)
		       "aaxxaa"))
(should-be-true (equal (nsubstitute #\x #\a "aaaaaa" :start 1 :end 4 :count 5)
		       "axxxaa"))

(should-be-true (equal (nsubstitute 'x 'y '((x) (y) (z)) :key #'car)
		       '((x) x (z))))
(should-be-true (eqv (nsubstitute 'x 'y #((x) (y) (z)) :key #'car)
		       #((x) x (z))))
(should-be-true (equal (nsubstitute #\x (char-int #\y) "xyz" :key #'char-int)
		       "xxz"))
(should-be-true (equal (nsubstitute-if 'x #'oddp '((1) (2) (3) (4) (5)) 
				  :key #'car) '(x (2) x (4) x)))
(should-be-true (equal (nsubstitute-if-not 'x #'oddp '((1) (2) (3) (4) (5))
				      :key #'car) '((1) x (3) x (5))))
(should-be-true (eqv (nsubstitute-if 'x #'oddp #((1) (2) (3) (4) (5)) 
				  :key #'car) #(x (2) x (4) x)))
(should-be-true (eqv (nsubstitute-if-not 'x #'oddp #((1) (2) (3) (4) (5))
				      :key #'car) #((1) x (3) x (5))))
(should-be-true (equal (nsubstitute-if #\x #'oddp "abcde" :key #'char-int)
		       "xbxdx"))
(should-be-true (equal (nsubstitute-if-not #\x #'oddp "abcde" :key #'char-int)
		       "axcxe"))
(should-be-true (equal (nsubstitute 'x '(b) '((a) (b) (c)) :test #'equal)
		       '((a) x (c))))
(should-be-true (equal (nsubstitute 'x '(b) '((a) (b) (c)) :test-not #'equal)
		       '(x (b) x)))

(should-fail (nsubstitute 'a 'b '(a b c) :count 'x))

(should-be-true (equal (substitute 'x 'b '()) '()))
(should-be-true (equal (substitute 'x 'b '(a b a b a)) '(a x a x a)))
(should-be-true (equal (substitute 'x 'a '(a b a b a)) '(x b x b x)))
(should-be-true (equal (substitute 'x 'b '(a b a b a) :start 1 :end 4)
		       '(a x a x a)))
(should-be-true (equal (substitute 'x 'a '(a b a b a) :start 1 :end 4)
		       '(a b x b a)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count -1)
		       '(b a c b d b)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 0)
		       '(b a c b d b)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 1 :from-end nil)
		       '(x a c b d b)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 2)
		       '(x a c x d b)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 3)
		       '(x a c x d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 4)
		       '(x a c x d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count nil)
		       '(x a c x d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 0 :from-end t)
		       '(b a c b d b)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 1 :from-end t)
		       '(b a c b d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 2 :from-end t)
		       '(b a c x d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 3 :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count 4 :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :count nil :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (substitute 'x 'b '(b a c b d b) :from-end t)
		       '(x a c x d x)))
(should-be-true (equal (substitute 'x 'a '(a a a a a a) :start 1 :end 4 :count 2)
		       '(a x x a a a)))
(should-be-true (equal (substitute 'x 'a '(a a a a a a) :start 1 :end 4 :count 5)
		       '(a x x x a a)))
(should-be-true (equal (substitute 'x 'a '(a a a a a a)
				    :start 1 :end 4 :count 2 :from-end t)
		       '(a a x x a a)))
(should-be-true (equal (substitute 'x 'a '(a a a a a a) :start 1 :end 4 :count 5)
		       '(a x x x a a)))

(should-be-true (eqv (substitute 'x 'b #()) #()))
(should-be-true (eqv (substitute 'x 'b #(a b a b a)) #(a x a x a)))
(should-be-true (eqv (substitute 'x 'a #(a b a b a)) #(x b x b x)))
(should-be-true (eqv (substitute 'x 'b #(a b a b a) :start 1 :end 4)
		       #(a x a x a)))
(should-be-true (eqv (substitute 'x 'a #(a b a b a) :start 1 :end 4)
		       #(a b x b a)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count -1)
		       #(b a c b d b)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 0)
		       #(b a c b d b)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 1 :from-end nil)
		       #(x a c b d b)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 2)
		       #(x a c x d b)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 3)
		       #(x a c x d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 4)
		       #(x a c x d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count nil)
		       #(x a c x d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 0 :from-end t)
		       #(b a c b d b)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 1 :from-end t)
		       #(b a c b d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 2 :from-end t)
		       #(b a c x d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 3 :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count 4 :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :count nil :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (substitute 'x 'b #(b a c b d b) :from-end t)
		       #(x a c x d x)))
(should-be-true (eqv (substitute 'x 'a #(a a a a a a) :start 1 :end 4 :count 2)
		       #(a x x a a a)))
(should-be-true (eqv (substitute 'x 'a #(a a a a a a) :start 1 :end 4 :count 5)
		       #(a x x x a a)))
(should-be-true (eqv (substitute 'x 'a #(a a a a a a)
				    :start 1 :end 4 :count 2 :from-end t)
		       #(a a x x a a)))
(should-be-true (eqv (substitute 'x 'a #(a a a a a a) :start 1 :end 4 :count 5)
		       #(a x x x a a)))

(should-be-true (equal (substitute #\x #\b "") ""))
(should-be-true (equal (substitute #\x #\b "ababa") "axaxa"))
(should-be-true (equal (substitute #\x #\a "ababa") "xbxbx"))
(should-be-true (equal (substitute #\x #\b "ababa" :start 1 :end 4)
		       "axaxa"))
(should-be-true (equal (substitute #\x #\a "ababa" :start 1 :end 4)
		       "abxba"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count -1)
		       "bacbdb"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 0)
		       "bacbdb"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 1 :from-end nil)
		       "xacbdb"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 2)
		       "xacxdb"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 3)
		       "xacxdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 4)
		       "xacxdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count nil)
		       "xacxdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 0 :from-end t)
		       "bacbdb"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 1 :from-end t)
		       "bacbdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 2 :from-end t)
		       "bacxdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 3 :from-end t)
		       "xacxdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count 4 :from-end t)
		       "xacxdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :count nil :from-end t)
		       "xacxdx"))
(should-be-true (equal (substitute #\x #\b "bacbdb" :from-end t)
		       "xacxdx"))
(should-be-true (equal (substitute #\x #\a "aaaaaa" :start 1 :end 4 :count 2)
		       "axxaaa"))
(should-be-true (equal (substitute #\x #\a "aaaaaa" :start 1 :end 4 :count 5)
		       "axxxaa"))
(should-be-true (equal (substitute #\x #\a "aaaaaa"
				    :start 1 :end 4 :count 2 :from-end t)
		       "aaxxaa"))
(should-be-true (equal (substitute #\x #\a "aaaaaa" :start 1 :end 4 :count 5)
		       "axxxaa"))

(should-be-true (equal (substitute 'x 'y '((x) (y) (z)) :key #'car)
		       '((x) x (z))))
(should-be-true (eqv (substitute 'x 'y #((x) (y) (z)) :key #'car)
		       #((x) x (z))))
(should-be-true (equal (substitute #\x (char-int #\y) "xyz" :key #'char-int)
		       "xxz"))
(should-be-true (equal (substitute-if 'x #'oddp '((1) (2) (3) (4) (5)) 
				  :key #'car) '(x (2) x (4) x)))
(should-be-true (equal (substitute-if-not 'x #'oddp '((1) (2) (3) (4) (5))
				      :key #'car) '((1) x (3) x (5))))
(should-be-true (eqv (substitute-if 'x #'oddp #((1) (2) (3) (4) (5)) 
				  :key #'car) #(x (2) x (4) x)))
(should-be-true (eqv (substitute-if-not 'x #'oddp #((1) (2) (3) (4) (5))
				      :key #'car) #((1) x (3) x (5))))
(should-be-true (equal (substitute-if #\x #'oddp "abcde" :key #'char-int)
		       "xbxdx"))
(should-be-true (equal (substitute-if-not #\x #'oddp "abcde" :key #'char-int)
		       "axcxe"))
(should-be-true (equal (substitute 'x '(b) '((a) (b) (c)) :test #'equal)
		       '((a) x (c))))
(should-be-true (equal (substitute 'x '(b) '((a) (b) (c)) :test-not #'equal)
		       '(x (b) x)))

(should-fail (substitute 'a 'b '(a b c) :count 'x))


;; Remove and delete (start and end keywords checked by "search"-- same code)

(should-be-true (equal (remove 'b '()) '()))
(should-be-true (equal (remove 'b '(a b a b a)) '(a a a)))
(should-be-true (equal (remove 'a '(a b a b a)) '(b b)))
(should-be-true (equal (remove 'b '(a b a b a) :start 1 :end 4) '(a a a)))
(should-be-true (equal (remove 'a '(a b a b a) :start 1 :end 4) '(a b b a)))
(should-be-true (equal (remove 'b '(b a c b d b) :count -1) '(b a c b d b)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 0) '(b a c b d b)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 1 :from-end nil)
		       '(a c b d b)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 2) '(a c d b)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 3) '(a c d)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 4) '(a c d)))
(should-be-true (equal (remove 'b '(b a c b d b) :count nil) '(a c d)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 0 :from-end t)
		       '(b a c b d b)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 1 :from-end t)
		       '(b a c b d)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 2 :from-end t)
		       '(b a c d)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 3 :from-end t)
		       '(a c d)))
(should-be-true (equal (remove 'b '(b a c b d b) :count 4 :from-end t)
		       '(a c d)))
(should-be-true (equal (remove 'b '(b a c b d b) :count nil :from-end t)
		       '(a c d)))
(should-be-true (equal (remove 'b '(b a c b d b) :from-end t) '(a c d)))
(should-be-true (equal (remove 'b '(a b a b a) :start 1 :end 4
			       :count 5 :from-end t) '(a a a)))
(should-be-true (equal (remove 'a '(a b a b a) :start 1 :end 4
			       :count 5 :from-end t) '(a b b a)))

(should-be-true (eqv (remove 'b #()) #()))
(should-be-true (eqv (remove 'b #(a b a b a)) #(a a a)))
(should-be-true (eqv (remove 'a #(a b a b a)) #(b b)))
(should-be-true (eqv (remove 'b #(a b a b a) :start 1 :end 4) #(a a a)))
(should-be-true (eqv (remove 'a #(a b a b a) :start 1 :end 4) #(a b b a)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count -1) #(b a c b d b)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 0) #(b a c b d b)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 1 :from-end nil)
		       #(a c b d b)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 2) #(a c d b)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 3) #(a c d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 4) #(a c d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count nil) #(a c d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 0 :from-end t)
		       #(b a c b d b)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 1 :from-end t)
		       #(b a c b d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 2 :from-end t)
		       #(b a c d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 3 :from-end t)
		       #(a c d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count 4 :from-end t)
		       #(a c d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :count nil :from-end t)
		       #(a c d)))
(should-be-true (eqv (remove 'b #(b a c b d b) :from-end t) #(a c d)))
(should-be-true (eqv (remove 'b #(a b a b a) :start 1 :end 4
			     :count 5 :from-end t) #(a a a)))
(should-be-true (eqv (remove 'a #(a b a b a) :start 1 :end 4
			     :count 5 :from-end t) #(a b b a)))

(should-be-true (equal (remove #\b "") ""))
(should-be-true (equal (remove #\b "ababa") "aaa"))
(should-be-true (equal (remove #\a "ababa") "bb"))
(should-be-true (equal (remove #\b "ababa" :start 1 :end 4) "aaa"))
(should-be-true (equal (remove #\a "ababa" :start 1 :end 4) "abba"))
(should-be-true (equal (remove #\b "bacbdb" :count -1) "bacbdb"))
(should-be-true (equal (remove #\b "bacbdb" :count 0) "bacbdb"))
(should-be-true (equal (remove #\b "bacbdb" :count 1 :from-end nil) "acbdb"))
(should-be-true (equal (remove #\b "bacbdb" :count 2) "acdb"))
(should-be-true (equal (remove #\b "bacbdb" :count 3) "acd"))
(should-be-true (equal (remove #\b "bacbdb" :count 4) "acd"))
(should-be-true (equal (remove #\b "bacbdb" :count nil) "acd"))
(should-be-true (equal (remove #\b "bacbdb" :count 0 :from-end t) "bacbdb"))
(should-be-true (equal (remove #\b "bacbdb" :count 1 :from-end t) "bacbd"))
(should-be-true (equal (remove #\b "bacbdb" :count 2 :from-end t) "bacd"))
(should-be-true (equal (remove #\b "bacbdb" :count 3 :from-end t) "acd"))
(should-be-true (equal (remove #\b "bacbdb" :count 4 :from-end t) "acd"))
(should-be-true (equal (remove #\b "bacbdb" :count nil :from-end t) "acd"))
(should-be-true (equal (remove #\b "bacbdb" :from-end t) "acd"))
(should-be-true (equal (remove #\b "ababa" :start 1 :end 4
			       :count 5 :from-end t) "aaa"))
(should-be-true (equal (remove #\a "ababa" :start 1 :end 4
			       :count 5 :from-end t) "abba"))

(should-be-true (equal (remove 'a '((a) (b) (a) (b) (a))
			       :key #'car) '((b) (b))))
(should-be-true (equal (remove '(a) '((a) (b) (a) (b) (a))
			       :test #'equal) '((b) (b))))
(should-be-true (equal (remove 'a '(a b a b a)
			       :test-not #'eq) '(a a a)))

(should-be-true (equal (remove 'y '((x) (y) (z)) :key #'car)
		       '((x) (z))))
(should-be-true (eqv (remove 'y #((x) (y) (z)) :key #'car)
		       #((x) (z))))
(should-be-true (equal (remove (char-int #\y) "xyz" :key #'char-int)
		       "xz"))
(should-be-true (equal (remove-if #'oddp '((1) (2) (3) (4) (5)) 
				  :key #'car) '((2) (4))))
(should-be-true (equal (remove-if-not #'oddp '((1) (2) (3) (4) (5))
				      :key #'car) '((1) (3) (5))))
(should-be-true (eqv (remove-if #'oddp #((1) (2) (3) (4) (5)) 
				  :key #'car) #((2) (4))))
(should-be-true (eqv (remove-if-not #'oddp #((1) (2) (3) (4) (5))
				      :key #'car) #((1) (3) (5))))
(should-be-true (equal (remove-if #'oddp "abcde" :key #'char-int)
		       "bd"))
(should-be-true (equal (remove-if-not #'oddp "abcde" :key #'char-int)
		       "ace"))
(should-be-true (equal (remove '(b) '((a) (b) (c)) :test #'equal)
		       '((a) (c))))
(should-be-true (equal (remove '(b) '((a) (b) (c)) :test-not #'equal)
		       '((b))))


(should-fail (delete 'b '(a b c) :count 'x))
(should-fail (remove 'b '(a b c) :count 'x))

(should-be-true (equal (delete 'b '()) '()))
(should-be-true (equal (delete 'b '(a b a b a)) '(a a a)))
(should-be-true (equal (delete 'a '(a b a b a)) '(b b)))
(should-be-true (equal (delete 'b '(a b a b a) :start 1 :end 4) '(a a a)))
(should-be-true (equal (delete 'a '(a b a b a) :start 1 :end 4) '(a b b a)))
(should-be-true (equal (delete 'b '(b a c b d b) :count -1) '(b a c b d b)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 0) '(b a c b d b)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 1 :from-end nil)
		       '(a c b d b)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 2) '(a c d b)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 3) '(a c d)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 4) '(a c d)))
(should-be-true (equal (delete 'b '(b a c b d b) :count nil) '(a c d)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 0 :from-end t)
		       '(b a c b d b)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 1 :from-end t)
		       '(b a c b d)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 2 :from-end t)
		       '(b a c d)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 3 :from-end t)
		       '(a c d)))
(should-be-true (equal (delete 'b '(b a c b d b) :count 4 :from-end t)
		       '(a c d)))
(should-be-true (equal (delete 'b '(b a c b d b) :count nil :from-end t)
		       '(a c d)))
(should-be-true (equal (delete 'b '(b a c b d b) :from-end t) '(a c d)))
(should-be-true (equal (delete 'b '(a b a b a) :start 1 :end 4
			       :count 5 :from-end t) '(a a a)))
(should-be-true (equal (delete 'a '(a b a b a) :start 1 :end 4
			       :count 5 :from-end t) '(a b b a)))

(should-be-true (eqv (delete 'b #()) #()))
(should-be-true (eqv (delete 'b #(a b a b a)) #(a a a)))
(should-be-true (eqv (delete 'a #(a b a b a)) #(b b)))
(should-be-true (eqv (delete 'b #(a b a b a) :start 1 :end 4) #(a a a)))
(should-be-true (eqv (delete 'a #(a b a b a) :start 1 :end 4) #(a b b a)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count -1) #(b a c b d b)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 0) #(b a c b d b)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 1 :from-end nil)
		       #(a c b d b)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 2) #(a c d b)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 3) #(a c d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 4) #(a c d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count nil) #(a c d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 0 :from-end t)
		       #(b a c b d b)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 1 :from-end t)
		       #(b a c b d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 2 :from-end t)
		       #(b a c d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 3 :from-end t)
		       #(a c d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count 4 :from-end t)
		       #(a c d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :count nil :from-end t)
		       #(a c d)))
(should-be-true (eqv (delete 'b #(b a c b d b) :from-end t) #(a c d)))
(should-be-true (eqv (delete 'b #(a b a b a) :start 1 :end 4
			     :count 5 :from-end t) #(a a a)))
(should-be-true (eqv (delete 'a #(a b a b a) :start 1 :end 4
			     :count 5 :from-end t) #(a b b a)))

(should-be-true (equal (delete #\b "") ""))
(should-be-true (equal (delete #\b "ababa") "aaa"))
(should-be-true (equal (delete #\a "ababa") "bb"))
(should-be-true (equal (delete #\b "ababa" :start 1 :end 4) "aaa"))
(should-be-true (equal (delete #\a "ababa" :start 1 :end 4) "abba"))
(should-be-true (equal (delete #\b "bacbdb" :count -1) "bacbdb"))
(should-be-true (equal (delete #\b "bacbdb" :count 0) "bacbdb"))
(should-be-true (equal (delete #\b "bacbdb" :count 1 :from-end nil) "acbdb"))
(should-be-true (equal (delete #\b "bacbdb" :count 2) "acdb"))
(should-be-true (equal (delete #\b "bacbdb" :count 3) "acd"))
(should-be-true (equal (delete #\b "bacbdb" :count 4) "acd"))
(should-be-true (equal (delete #\b "bacbdb" :count nil) "acd"))
(should-be-true (equal (delete #\b "bacbdb" :count 0 :from-end t) "bacbdb"))
(should-be-true (equal (delete #\b "bacbdb" :count 1 :from-end t) "bacbd"))
(should-be-true (equal (delete #\b "bacbdb" :count 2 :from-end t) "bacd"))
(should-be-true (equal (delete #\b "bacbdb" :count 3 :from-end t) "acd"))
(should-be-true (equal (delete #\b "bacbdb" :count 4 :from-end t) "acd"))
(should-be-true (equal (delete #\b "bacbdb" :count nil :from-end t) "acd"))
(should-be-true (equal (delete #\b "bacbdb" :from-end t) "acd"))
(should-be-true (equal (delete #\b "ababa" :start 1 :end 4 :count 5
			       :from-end t) "aaa"))
(should-be-true (equal (delete #\a "ababa" :start 1 :end 4 :count 5
			       :from-end t) "abba"))

(should-be-true (equal (delete 'y '((x) (y) (z)) :key #'car)
		       '((x) (z))))
(should-be-true (eqv (delete 'y #((x) (y) (z)) :key #'car)
		       #((x) (z))))
(should-be-true (equal (delete (char-int #\y) "xyz" :key #'char-int)
		       "xz"))
(should-be-true (equal (delete 'a '((a) (b) (a) (b) (a))
			       :key #'car) '((b) (b))))
(should-be-true (equal (delete '(a) '((a) (b) (a) (b) (a))
			       :test #'equal) '((b) (b))))
(should-be-true (equal (delete 'a '(a b a b a)
			       :test-not #'eq) '(a a a)))

(should-be-true (equal (delete-if #'oddp '((1) (2) (3) (4) (5)) 
				  :key #'car) '((2) (4))))
(should-be-true (equal (delete-if-not #'oddp '((1) (2) (3) (4) (5))
				      :key #'car) '((1) (3) (5))))
(should-be-true (eqv (delete-if #'oddp #((1) (2) (3) (4) (5)) 
				  :key #'car) #((2) (4))))
(should-be-true (eqv (delete-if-not #'oddp #((1) (2) (3) (4) (5))
				      :key #'car) #((1) (3) (5))))
(should-be-true (equal (delete-if #'oddp "abcde" :key #'char-int)
		       "bd"))
(should-be-true (equal (delete-if-not #'oddp "abcde" :key #'char-int)
		       "ace"))
(should-be-true (equal (remove '(b) '((a) (b) (c)) :test #'equal)
		       '((a) (c))))
(should-be-true (equal (remove '(b) '((a) (b) (c)) :test-not #'equal)
		       '((b))))


;; COUNT-IF

(should-be-true (eql (count-if #'oddp '()) 0))
(should-be-true (eql (count-if #'oddp '(1 2 3 4 5)) 3))
(should-be-true (eql (count-if #'oddp '(1 2 3 4 5) :from-end t) 3))
(should-be-true (eql (count-if #'oddp '(1 2 3 4 5) :start 1 :end 4) 1))
(should-be-true (eql (count-if #'oddp '((1) (2) (3) (4) (5)) :key #'car) 3))
(should-be-true (eql (count-if #'oddp #()) 0))
(should-be-true (eql (count-if #'oddp #(1 2 3 4 5)) 3))
(should-be-true (eql (count-if #'oddp #(1 2 3 4 5) :from-end t) 3))
(should-be-true (eql (count-if #'oddp #(1 2 3 4 5) :start 1 :end 4) 1))
(should-be-true (eql (count-if #'oddp #((1) (2) (3) (4) (5)) :key #'car) 3))
(should-be-true (eql (count-if #'oddp "" :key #'char-int) 0))
(should-be-true (eql (count-if #'oddp "abcde" :key #'char-int) 3))
(should-be-true (eql (count-if #'oddp "abcde" :key #'char-int 
			       :start 1 :end 4) 1))
(should-be-true (eql (count-if-not #'oddp '(1 2 3 4 5)) 2))
(should-be-true (eql (count 2 '((1) (2) (3) (2) (5)) :key #'car) 2))
(should-be-true (eql (count 2 '((1) (2) (3) (4) (5)) :test #'< :key #'car) 3))
(should-be-true (eql (count 2 '((1) (2) (3) (4) (5)) :test-not #'< :key #'car) 2))

;; FIND-IF
(should-be-true (eql (find-if #'oddp '()) nil))
(should-be-true (eql (find-if #'oddp '(1 2 3 4 5)) 1))
(should-be-true (eql (find-if #'oddp '(2 4 5)) 5))
(should-be-true (eql (find-if #'oddp '(1 2 3 4 5) :start 1 :end 4) 3))
(should-be-true (equal (find-if #'oddp '((1) (2) (3) (4) (5)) :key #'car) '(1)))
(should-be-true (eql (find-if #'oddp '(1 3 5 7) :from-end t) 7))
(should-be-true (eql (find-if #'evenp '(4 2 5 7) :from-end t) 2))

(should-be-true (eql (find-if #'oddp #()) nil))
(should-be-true (eql (find-if #'oddp #(1 2 3 4 5)) 1))
(should-be-true (eql (find-if #'oddp #(2 4 5)) 5))
(should-be-true (eql (find-if #'oddp #(1 2 3 4 5) :start 1 :end 4) 3))
(should-be-true (equal (find-if #'oddp #((1) (2) (3) (4) (5)) :key #'car) '(1)))
(should-be-true (eql (find-if #'oddp #(1 3 5 7) :from-end t) 7))
(should-be-true (eql (find-if #'evenp #(4 2 5 7) :from-end t) 2))

(should-be-true (eql (find-if #'oddp "" :key #'char-int) nil))
(should-be-true (eql (find-if #'oddp "abcde" :key #'char-int) #\a))
(should-be-true (eql (find-if #'oddp "abcde" :key #'char-int 
			       :start 1 :end 4) #\c))
(should-be-true (eq (find-if #'evenp "acebd" :key #'char-int :from-end t) #\d))
(should-be-true (eq (find-if #'oddp "abcde" :key #'char-int :from-end t) #\e))
(should-be-true (eq (find-if #'evenp "dbace" :key #'char-int :from-end t) #\b))

(should-be-true (eql (find-if-not #'oddp '(1 2 3 4 5)) 2))
(should-be-true (equal (find 3 '((1) (2) (3 x) (4)) :key #'car) '(3 x)))

;; POSITION-IF
(should-be-true (eql (position-if #'oddp '()) nil))
(should-be-true (eql (position-if #'oddp '(1 2 3 4 5)) 0))
(should-be-true (eql (position-if #'oddp '(1 2 3 4 5) :from-end t) 4))
(should-be-true (eql (position-if #'oddp '(1 2 3 4 5) :start 2 :end 4) 2))
(should-be-true (eql (position-if #'oddp '(1 2 3 4 5) :start 3 :end 4) nil))
(should-be-true (eql (position-if #'oddp '(1 2 3 4 5 6 7)
				  :start 1 :end 5 :from-end t) 4))
(should-be-true (eql (position-if #'oddp '(1 2 3 4 5 6 7)
				  :start 1 :end 4 :from-end t) 2))
(should-be-true (eql (position-if #'oddp '((1) (2) (3) (4) (5)) :key #'car) 0))
(should-be-true (eql (position-if #'oddp #()) nil))
(should-be-true (eql (position-if #'oddp #(1 2 3 4 5)) 0))
(should-be-true (eql (position-if #'oddp #(1 2 3 4 5) :from-end t) 4))
(should-be-true (eql (position-if #'oddp #(1 2 3 4 5) :start 1 :end 4) 2))
(should-be-true (eql (position-if #'oddp #(1 2 3 4 5 6 7)
				  :start 1 :end 5 :from-end t) 4))
(should-be-true (eql (position-if #'oddp #(1 2 3 4 5 6 7)
				  :start 1 :end 4 :from-end t) 2))
(should-be-true (eql (position-if #'oddp #((1) (2) (3) (4) (5)) :key #'car) 0))
(should-be-true (eql (position-if #'oddp "" :key #'char-int) nil))
(should-be-true (eql (position-if #'oddp "abcde" :key #'char-int) 0))
(should-be-true (eql (position-if #'oddp "abcde" :key #'char-int :from-end t)
		     4))
(should-be-true (eql (position-if #'oddp "abcde" :key #'char-int 
			       :start 1 :end 4) 2))
(should-be-true (eql (position-if-not #'oddp '(1 2 3 4 5)) 1))
(should-be-true (eql (position 3 '((1) (2) (3 x) (4)) :key #'car) 2))

;; REDUCE
(should-fail (reduce #'(lambda (x y) (if (char< x y) y x)) ""))
(should-be-true (eq (reduce #'(lambda (x y) (if (char< x y) y x)) ""
			    :initial-value #\c) #\c))
(should-be-true (eq (reduce #'(lambda (x y) (if (char< x y) y x)) "c") #\c))
(should-be-true (eq (reduce #'(lambda (x y) (if (char< x y) y x)) "x"
			    :initial-value #\space) #\x))
(should-be-true (eq (reduce #'(lambda (x y) (if (char< x y) y x)) "aubc"
			    :initial-value #\space) #\u))
(should-be-true (eq (reduce #'(lambda (x y) (if (char< x y) y x)) "uabcx"
			    :initial-value #\space :start 1 :end 4) #\c))

(should-be-true (eql (reduce #'+ '()) 0))
(should-be-true (eql (reduce #'car '() :initial-value 4) 4))
(should-be-true (eql (reduce #'car '(3)) 3))
(should-be-true (eql (reduce #'car '(4) :initial-value 3 :start 1) 3))
(should-be-true (eql (reduce #'+ '(2 3) :initial-value 1) 6))
(should-be-true (eql (reduce #'+ '(2 3 4 5) :initial-value 1
			     :start 1 :end 3) 8))

(should-be-true (eql (reduce #'+ #()) 0))
(should-be-true (eql (reduce #'car #() :initial-value 4) 4))
(should-be-true (eql (reduce #'car #(3)) 3))
(should-be-true (eql (reduce #'car #(4) :initial-value 3 :start 1) 3))
(should-be-true (eql (reduce #'+ #(2 3) :initial-value 1) 6))
(should-be-true (eql (reduce #'+ #(2 3 4 5) :initial-value 1
			     :start 1 :end 3) 8))


(terpri)

(princ test-number)
(princ " tests performed.")
(terpri)

(princ bugs-found)
(princ " bugs found.")
(terpri)
