package sudoku;

import javax.swing.JComponent;
import java.awt.*;
import java.awt.event.*;

/**
 * Component that displays the playing field. This is actually the most
 * difficult part of the design, so I took lots of it from the 68HCS12 simulator
 * code!
 * @author: Tom Almy
 */
public class ViewScreen extends JComponent implements java.awt.event.FocusListener, MouseListener {

    static final private int PHYS_TO_LOG[] = {0, 1, 2, 3, 3, 4, 5, 6, 6, 7, 8};
    static final private int LOG_TO_PHYS[] = {0, 1, 2, 4, 5, 6, 8, 9, 10};
    static final private int COLUMNS = 11;
    static final private int ROWS = 11;
    private int _charWidth, _ascent, _descent, _charHeight, _leading;
    private Dimension _minimumSize = new Dimension(0, 0);
    private Image _offscreen;
    private int _curx = 0; // Physical location
    private int _cury = 0; // Physical location
    private boolean _curVis = false;
    private boolean _gotFocus = false;
    private java.awt.Color _controlColor;
    private java.awt.Color _controlTextColor;
    private java.awt.Color _controlLockedColor;
    private boolean _hinting = false;

    /**
     * Turns on or off hinting background
     * @param value
     */
    public void setHinting(boolean value) {
        _hinting = value;
    }

    /**
     * Construct our screen
     */
    public ViewScreen() {
        setSize(560, 375);
        // create a panel just to get default font size and colors
        int fontSize;
        {
            javax.swing.JPanel temp = new javax.swing.JPanel();
            fontSize = (int) (temp.getFont().getSize() * 1.5); // Want bigger than normal
            _controlColor = Color.GRAY;
            _controlTextColor = temp.getForeground(); // Only color we actually get from panel
            _controlLockedColor = Color.PINK;
        }
        Font f = new Font("Monospaced", Font.BOLD, fontSize);
        setFont(f);
        FontMetrics fm = getFontMetrics(f);
        _ascent = fm.getAscent();
        _descent = fm.getDescent();
        _leading = fm.getLeading();
        _charHeight = fm.getHeight() + 2;
        _charWidth = fm.charWidth('A') * 2;
        int desiredHeight = ROWS * _charHeight;
        int desiredWidth = COLUMNS * _charWidth;
        setSize(desiredWidth, desiredHeight);
        _minimumSize = new Dimension(desiredWidth, desiredHeight);
        addFocusListener(this);
        addMouseListener(this);
        setRequestFocusEnabled(true);
    }

    /**
     * Make cursor invisible.
     * Creation date: (8/10/2000 6:51:08 PM)
     * @param g java.awt.Graphics
     */
    private void cursorOff(Graphics g) {
        // turn the cursor off
        if (_curVis) {
            _curVis = false;
            g.setColor(getBackground());
            g.setXORMode(Color.red);
            g.fillRect(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            repaint(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            g.setPaintMode();
            g.setColor(getForeground());
        }
    }

    /**
     * Make the cursor visible.
     * Creation date: (8/10/2000 6:51:08 PM)
     * @param g java.awt.Graphics
     */
    private void cursorOn(Graphics g) {
        // turn the cursor on
        if (_gotFocus) {
            _curVis = true;
            g.setColor(getBackground());
            g.setXORMode(Color.red);
            g.fillRect(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            repaint(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            g.setPaintMode();
            g.setColor(getForeground());
        }
    }

    /**
     * We have received focus. Draw the cursor
     *
     * @param e FocusEvent
     */
    public void focusGained(java.awt.event.FocusEvent e) {
        if (_offscreen == null) {
            return; // Nothing to do yet
        }
        if (!_gotFocus) {
            _gotFocus = true;
            if (_curVis) {
                return;
            }
            _curVis = true;
            Graphics g = _offscreen.getGraphics();
            g.setColor(getBackground());
            g.setXORMode(Color.red);
            g.fillRect(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            repaint(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            g.setPaintMode();
            g.setColor(getForeground());
            g.dispose();
        }

    }

    /**
     * We have lost focus. Erase the cursor.
     *
     * @param e FocusEvent
     */
    public void focusLost(java.awt.event.FocusEvent e) {
        if (_gotFocus) {
            _gotFocus = false;
            if (!_curVis) {
                return;
            }
            _curVis = false;
            Graphics g = _offscreen.getGraphics();
            g.setColor(getBackground());
            g.setXORMode(Color.red);
            g.fillRect(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            repaint(_curx * _charWidth, _cury * _charHeight, _charWidth, _charHeight);
            g.setPaintMode();
            g.setColor(getForeground());
            paint(g);
            g.dispose();
        }
    }

    /**
     * Convert a pixel position to a character coordinate
     * @return java.awt.Point
     * @param p java.awt.Point
     */
    private Point getCoordinates(Point p) {
        int row = p.x / _charWidth;
        int col = p.y / _charHeight;
        return new Point(row, col);
    }

    /**
     * Get the current cursor position in logical coordinates
     * Creation date: (8/10/2000 7:00:58 PM)
     * @return java.awt.Point
     */
    public Point getLogCursorPosition() {
        return new Point(PHYS_TO_LOG[_curx], PHYS_TO_LOG[_cury]);
    }

    /**
     * Gets the mininimum size of this component.
     * @return A dimension object indicating this component's minimum size.
     * @see #getPreferredSize
     * @see java.awtLayoutManager
     */
    @Override
    public Dimension getMinimumSize() {
        return _minimumSize;
    }

    /**
     * Gets the preferred size of this component.
     * @return A dimension object indicating this component's preferred size.
     * @see #getMinimumSize
     * @see java.awt.LayoutManager
     */
    @Override
    public Dimension getPreferredSize() {
        return _minimumSize;
    }

    @Override
    public boolean isFocusable() {
        return true;
    }

    /**
     * Generated the offscreen image so we can buffer the display content for speed.
     * Creation date: (8/5/2000 4:15:55 PM)
     */
    private void makeScreen() {
        if (_offscreen != null) {
            return;
        }
        int desiredWidth = _minimumSize.width;
        int desiredHeight = _minimumSize.height;

        Frame frame = new Frame(); // Tempoarary frame so we can create the image
        frame.addNotify();  // I don't know why
        frame.setFont(this.getFont());

        _offscreen = frame.createImage(desiredWidth, desiredHeight);
        Graphics g = _offscreen.getGraphics();

        // Draw the backgroun
        g.setColor(getBackground());
        g.fillRect(0, 0, desiredWidth, desiredHeight);

        // Draw background grid
        g.setColor(_controlColor);
        g.fillRect(0, 3 * _charHeight, _charWidth * COLUMNS, _charHeight);
        g.fillRect(0, 7 * _charHeight, _charWidth * COLUMNS, _charHeight);
        g.fillRect(3 * _charWidth, 0, _charWidth, _charHeight * ROWS);
        g.fillRect(7 * _charWidth, 0, _charWidth, _charHeight * ROWS);

        // turn the cursor on
        cursorOn(g);
        g.setColor(getForeground());
        g.dispose();
        frame.dispose();
        requestFocus();

    }

    /**
     * This method is called to repaint this component.
     *
     * @param g Graphics
     */
    @Override
    public void paintComponent(Graphics g) {
        if (_offscreen == null) {
            makeScreen();
        }
        g.drawImage(_offscreen, 0, 0, null);
    }

    /**
     * Display value at location
     * @param x
     * @param y
     * @param f Field object
     */
    public void putText(int x, int y, Field f) {
        if (_offscreen == null) {
            makeScreen();
        }
        Graphics g = _offscreen.getGraphics();

        cursorOff(g);

        int px = LOG_TO_PHYS[x];
        int py = LOG_TO_PHYS[y];
        int index = Field.coordinatesToIndex(x, y);

        // Must erase Background


        if (_hinting) {
            switch (f.hintValue(index)) {
                case NO_HINT:
                    g.setColor(getBackground());
                    break;
                case ONE_HINT:
                    g.setColor(Color.GREEN);
                    break;
                case TWO_HINT:
                    g.setColor(Color.YELLOW);
                    break;
            }
        } else {
            g.setColor(getBackground());
        }
        g.fillRect(px * _charWidth, py * _charHeight, _charWidth, _charHeight);

        // Now draw foreground
        if (f.isLocked(index)) {
            g.setColor(_controlLockedColor);
        } else {
            g.setColor(_controlTextColor);
        }
        g.drawString(f.squares[index] == 0 ? " " : Integer.toString(f.squares[index]),
                (int) (px * _charWidth + _charWidth * 0.3),
                py * _charHeight + _ascent);
        repaint(px * _charWidth, py * _charHeight, _charWidth, _charHeight);

        cursorOn(g);
        g.dispose();
    }

    /**
     * Redraw entire field
     * @param f field to draw
     */
    public void drawAll(Field f) {
        for (int i = 0; i < Field.LOCATIONS; i++) {
            int y = Field.indexToRow(i);
            int x = Field.indexToColumn(i);
            putText(x, y, f);
        }
    }

    /**
     * Position the cursor at the desired coordinates
     * Creation date: (8/10/2000 6:58:23 PM)
     * @param x int
     * @param y int
     */
    private void setCursorPosition(int x, int y) {
        if (x != _curx || y != _cury) {
            Graphics g = _offscreen.getGraphics();
            cursorOff(g);
            _curx = x;
            _cury = y;
            cursorOn(g);
            g.dispose();
        }
    }

    /**
     * Position the cursor at the desired logical coordinates
     * @param x
     * @param y
     */
    public void setLogCursorPosition(int x, int y) {
        setCursorPosition(LOG_TO_PHYS[x], LOG_TO_PHYS[y]);
    }

    /* Update does not clear background */
    @Override
    public void update(Graphics g) {
        paint(g);
    }

    public void mouseClicked(MouseEvent e) {
        // If the mouse is clicked, we set the cursor to the closest position
        java.awt.Point p = getCoordinates(e.getPoint());
        e.consume();
        requestFocus();
        int xPos = p.x;
        if (xPos == 3) {
            xPos = 4;
        } else if (xPos == 7) {
            xPos = 8;
        }
        int yPos = p.y;
        if (yPos == 3) {
            yPos = 4;
        } else if (yPos == 7) {
            yPos = 8;
        }
        setCursorPosition(xPos, yPos);
        return;

    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void mousePressed(MouseEvent e) {
        // I'm not sure about why this is needed
        e.consume();
        requestFocus();
    }

    public void mouseReleased(MouseEvent e) {
    }
}
